## ISTARF-CONUS (Inferred Storage Targets and Release Functions for Conterminous United States)
## Demo for converting parameters to policies

## Author: Sean Turner (sean.turner@pnnl.gov)
## Date created: 2021-03-25

# 1. Install `starfit` package and load required libraries

# devtools::install_github("IMMM-SFA/starfit")

library(starfit)
library(dplyr)
library(readr)
library(tidyr)
library(ggplot2)


# 2. Read ISTARF-CONUS and select example dam

read_csv("data/ISTARF/ISTARF-CONUS.csv") -> ISTARF_CONUS
# note: avoid using other file read pacakges, such as vroom, ...
# ... since this may fail to parse the Inf and -Inf values

ISTARF_CONUS %>%
  subset(GRanD_ID == 7318) ->
  #subset(GRanD_NAME == "Libby") ->
  params

# 3. Compute Normal Operating Range ("NOR") using `starfit` ...
# ... convert_parameters_to_targets() function.
# NOTE: parameters argument requires order: mu, alpha, beta, max, min

# upper bound of NOR
convert_parameters_to_targets(
  parameters = c(params[["NORhi_mu"]],
                 params[["NORhi_alpha"]],
                 params[["NORhi_beta"]],
                 params[["NORhi_max"]],
                 params[["NORhi_min"]]),
  # give the harmonic a name...
  target_name = "NORhi"
  ) -> NOR_upper_bound

# lower bound of NOR
convert_parameters_to_targets(
  parameters = c(params[["NORlo_mu"]],
                 params[["NORlo_alpha"]],
                 params[["NORlo_beta"]],
                 params[["NORlo_max"]],
                 params[["NORlo_min"]]),
  # give the harmonic a name...
  target_name = "NORlo"
) -> NOR_lower_bound

# combine upper and lower to single table and plot

NOR_upper_bound %>%
  left_join(NOR_lower_bound,
            by = "epiweek") ->
  NOR

NOR %>%
  ggplot(aes(x = epiweek)) +
  geom_ribbon(aes(ymin = NORlo, ymax = NORhi)) +
  ylim(0, 100)

# note that the storage targets are presented as % of capacity...
# ... so to make them operational we need to multiply by reservoir capacity...

NOR %>%
  mutate(capacity_MCM = params[["GRanD_CAP_MCM"]],
         NORhi_actual = capacity_MCM * NORhi / 100,
         NORlo_actual = capacity_MCM * NORlo / 100) ->
  NOR_MCM

# plot actual NOR
NOR_MCM %>%
  ggplot(aes(x = epiweek, y = capacity_MCM)) +
  geom_line(linetype = 2) +
  geom_ribbon(aes(ymin = NORlo_actual, ymax = NORhi_actual)) +
  expand_limits(y = 0)


# 4. Compute the release for a given time of year, storage, and inflow condition.
# For this example, let's suppose it's the first week of the hydrological year...
# ... (in this case early October ~ EPI week 40) and that storage is relatively...
# ... low for this time of year, at 5,000 MCM and that the forecasted inflow volume...
# ... for the coming *week* is 120 MCM (~200 m3/s average).

# First, it's important to note that ResOpsUS_IP provides operations at weekly resolution...
# and release is standardized with mean long-term inflow in units of MCM/week.

storage_MCM <- 5000
inflow_vol <- 120 # inflow volume for week 40

params[["GRanD_MEANFLOW_CUMECS"]] -> mean_flow_cumecs
mean_flow_cumecs * 0.6048 -> mean_flow_MCMpWK

# define the release harmonic
convert_parameters_to_release_harmonic(c(
  params$Release_alpha1,
  params$Release_beta1,
  params$Release_alpha2,
  params$Release_beta2)
  ) -> release_harmonic

# plot
release_harmonic %>%
  ggplot(aes(epiweek, release_harmonic)) +
  geom_vline(xintercept = 40, col = "red") +
  geom_line()
# we can see that the typical release pattern is for low release in week 40.


# This is refered to as the "unadjusted release" because it is based solely on time-of-year...
# ... and is not yet adjusted for current storage and flow conditions. The release adjustment...
# .. is calculated using the "c", "p1", and "p2" parameters. We need to first define...
# ... (a) the storage state, and (b) the inflow state.

# calculate storage state
NOR_MCM %>%
  filter(epiweek == 40) %>%
  mutate(current_storage = storage_MCM) %>%
  mutate(storage_state = (current_storage - NORlo_actual) / (NORhi_actual - NORlo_actual)) %>%
  .[["storage_state"]] -> storage_state

# calculate standardized inflow
# standardize inflow vol as relative diff. from mean
(inflow_vol - mean_flow_MCMpWK) / mean_flow_MCMpWK ->
  inflow_standardized

# calculate release adjustment...

params$Release_c +
  params$Release_p1 * storage_state +
  params$Release_p2 * inflow_standardized ->
  release_adjustment_standardized

# then add to the harmonic to get the final standardized release:

release_harmonic %>%
  filter(epiweek == 40) %>%
  .[["release_harmonic"]] +
  release_adjustment_standardized ->
  adjusted_release_standardized

# In this example case the adjustment is zero. When all parameters of the release adjustment...
# ... are zero, this indicates that no suitable release adjustmnet model was found. Release for...
# ... Libby is therefore represented solely according to the seasonal harmoinic.

# Finally, destandardize to get the weekly desired release in MCM.
(adjusted_release_standardized * mean_flow_MCMpWK) + mean_flow_MCMpWK ->
  target_release_MCM


# So the release implemented in simulation for this particular week would be...
# ... approximately 122 MCM, or just more than the inflow coming in for that week.

# We also want to make sure the release isn't constrained by maximum or minimum constraints...

(params[["Release_max"]] * mean_flow_MCMpWK) + mean_flow_MCMpWK # ~428 MCM max OK!
(params[["Release_min"]] * mean_flow_MCMpWK) + mean_flow_MCMpWK # ~69 MCM min OK!
