# Determine reservoir and stream reach for each hydropower plant

map_reservoir_and_reach <- function(EHA, HILARRI, HESC, COMID_corrections, target_plants){

  target_plants |> 
    left_join(EHA,
              by = c("EIA_ID" = "EIA_PtID")) |> 
    arrange(-CH_MW) |> 
    select(EIA_ID, Mode) |> 
    unique() |> 
    summarise(dominant_mode = first(Mode),
              .by = EIA_ID) ->
    plant_dominant_modes

  target_plants |> 
    left_join(HILARRI |> mutate(EIA_ID = as.integer(eia_ptid)),
              by = "EIA_ID") |> 
    select(EIA_ID, grand_id, nidid, nhdv2comid, COMPLXID = eha_cmplx, fcdocket, pt_name,
           huc_12, pt_lon, pt_lat, eha_ptid, Hyd_hd_est_m, usgs_gage) |> 
    unique() ->
    res_reach_targets_unfiltered
  
  # bring in hydraulic head estimates from HESC
  HESC |>
    select(NHYD_HT, NID_HT, G_HT, COMPLXID) |>
    filter(COMPLXID %in% res_reach_targets_unfiltered[["COMPLXID"]]) |>
    mutate(NHYD_HT = if_else(NHYD_HT > 0, NHYD_HT, NA_real_),
           NID_HT = if_else(NID_HT > 0, NID_HT, NA_real_),
           G_HT = if_else(G_HT > 0, G_HT, NA_real_)) |>
    filter(!is.na(COMPLXID)) |> 
    summarise(NHYD_HT = sum(NHYD_HT, na.rm = T),
              NID_HT = sum(NID_HT, na.rm = T),
              G_HT = sum(G_HT, na.rm = T),
              .by = COMPLXID) |>
    unique() ->
    HESC_ready

  res_reach_targets_unfiltered |> 
    count(EIA_ID) |> filter(n > 1) |> 
    pull(EIA_ID) -> duplicated_plants
  
  res_reach_targets_unfiltered |> 
    filter(EIA_ID %in% duplicated_plants) |>
    #
    filter(!(EIA_ID == 518 & nidid == "CO01654")) |> 
    # Flatiron power plant receives water from Pinewood, not Flatiron reservoir
    #
    filter(!(EIA_ID == 1574 & nidid == "MD00097")) |> 
    # Conowingo--take the version with downstream gauge
    #
    filter(!(EIA_ID == 2719 & nidid == "NC00141")) |> 
    # Bridgewater--no diff
    #
    filter(!(EIA_ID == 2750 & nidid == "NC00378")) |> 
    # Thorpe--no diff
    #
    filter(!(EIA_ID == 2981 & nhdv2comid == 21773239)) |> 
    # Pensacola--diff comids, neither on POI list
    #
    filter(!(EIA_ID == 3014 & nidid == "OR00258")) |> 
    # Oxbow--only diff is state (keep ID, drop OR)
    #
    filter(!(EIA_ID == 3084 & nidid == "OR00616")) |> 
    # McNary--only diff is state (keep WA, drop OR)
    #
    filter(!(EIA_ID == 3373 & nhdv2comid == 11554136)) |> 
    # Fort Randall--diff comids)
    #
    filter(!(EIA_ID == 4011 & nidid == "WI00023")) |> 
    # St Croix Falls--only diff is state (keep MN, drop WI)
    #
    filter(!(EIA_ID == 6527 & nhdv2comid == 22294702)) |> 
    # EJ West-- keep comid in POI
    #
    filter(!(EIA_ID == 6636 & nidid == "PA01939")) |> 
    # Lake Lynn--keep WV, drop PA
    #
    filter(!(EIA_ID == 7511 & nidid == "OR00616")) |> 
    # McNary Fish--keep WA, drop OR
    #
    filter(!(EIA_ID == 8225 & nidid == "PA00022")) |> 
    # Seneca--keep case with usgs identified
    #
    filter(!(EIA_ID == 10253 & fcdocket == "P-6061")) |> 
    # Take *lower* Haypass Creek
    #
    filter(!(EIA_ID == 10186 & nidid == "ME00086")) |> 
    # two facilities... use downstream (with GRanD ID)
    #
    filter(!(EIA_ID == 54134 & COMPLXID %in% c("hc1492", "hc1493", "hc1494",
                                               "hc1495", "hc1497"))) |> 
    # use downstream facility for reach ID
    #
    filter(!(EIA_ID ==54580 & nidid == "NY00138")) |> 
    # use downstream of two RoR facilities
    #
    filter(!(EIA_ID ==54639 & nidid %in% c("NH00088", "NH00089", "NH00158",
                                           "NH00163", "NH00191"))) |> 
    # use downstream facility COMID (cascade)
    #
    filter(!(EIA_ID == 57690 & COMPLXID == "hc0166")) ->
    filtered_duplicates
  
  # add missing names for two plants
  res_reach_targets_unfiltered |> 
    filter(!EIA_ID %in% duplicated_plants) |> 
    bind_rows(filtered_duplicates) ->
    # below corrected in HILARRI v3
    # mutate(pt_name = if_else(EIA_ID == 2685, "CRESCENT", pt_name),
    #        pt_name = if_else(EIA_ID == 7820, "BIG QUINNESEC 92", pt_name)) ->
    target_plants_mapped_to_water
  
  vroom(COMID_corrections, skip = 2) -> corrections
  
  target_plants_mapped_to_water |> 
    left_join(corrections, by = join_by(COMPLXID)) |> 
    mutate(nhdv2comid = if_else(!is.na(COMID_replacement),
                                COMID_replacement,
                                nhdv2comid)) |> 
    select(EIA_ID, COMPLXID, plant_name = pt_name, grand_id, nidid, COMID = nhdv2comid,
           huc_12, pt_lon, pt_lat, Hyd_hd_est_m, usgs_gage) ->
    target_plants_mapped_to_water_corrected
  
  
  target_plants_mapped_to_water_corrected |>
    mutate(COMID = as.integer(COMID)) |>
    left_join(HESC_ready, by = join_by(COMPLXID)) |>
    mutate(Hyd_hd_est_m = if_else(Hyd_hd_est_m == 0, NA_real_, Hyd_hd_est_m),
           NHYD_HT = if_else(NHYD_HT == 0, NA_real_, NHYD_HT),
           NID_HT = if_else(NID_HT == 0, NA_real_, NID_HT),
           G_HT = if_else(G_HT == 0, NA_real_, G_HT)) |>
    mutate(H_m = case_when(
      !is.na(NHYD_HT) ~ NHYD_HT,
      is.na(NHYD_HT) & !is.na(Hyd_hd_est_m) ~ Hyd_hd_est_m,
      is.na(NHYD_HT) & is.na(Hyd_hd_est_m) & !is.na(NID_HT) ~ NID_HT,
      TRUE ~ G_HT
    )) |> 
    select(EIA_ID, COMPLXID, plant_name, grand_id, nidid, COMID,
           huc_12, pt_lon, pt_lat, H_m, usgs_gage) ->
    target_plants_mapped_to_water_corrected_H
  
  
  # Some 16 reservoirs are linked to 2 separate EIA IDs...
  # ... (e.g., Hoover NV and Hoover AZ). These are all unique cases,...
  # ... requiring each one dealt with manually

  bind_rows(
    # 1. BALCH 1 and BALCH 2. Same dam, two powerhouses: COMBINE.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc1017") |> 
      mutate(RHPID = "217/218_BALCH 1&2"),
    # 2. CARIBOU 1 and CARIBOU 2. Same dam: COMBINE.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc1241") |> 
      mutate(RHPID = "222/223_CARIBOU 1&2"),
    # 3. DRUM 1 and DRUM 2. Same reservoir: COMBINE.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc1372") |> 
      mutate(RHPID = "235/236_DRUM 1 & 2"),
    # 4. HAAS and KINGS RIVER. Same reservoir: COMBINE.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc1172") |> 
      mutate(RHPID = "240/254_HAAS-KINGS RIVER PROJECT"),
    # 5. UPPER & MIDDLE GORGE SAME REACH OF OWENS RIVER: COMBINE.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc4017") |> 
      mutate(RHPID = "401/407_UPPER & MIDDLE GORGE"), #Fixing the RHPID to match EIA
    # 6. COLAGE / NARROWS 2. SAME RESOURCE: COMBINE. NEGLECT 3RD EIA_ID "FISH POWER" - JUST 0.2 MW.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc1337") |> 
      mutate(RHPID = "454/455_COLGATE & NARROWS 2"),
    # 7. WHITE ROCK / SLAB CREEK & JONES FORK. SAME RESOURCE: COMBINE. NEGLECT 3RD EIA_ID "SLAB CREEK" - JUST 0.5MW.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc1239") |> 
      mutate(RHPID = "435/534_WHITE ROCK/SLAB CREEK & JONES FORK") |> 
      # take properties for Jones Fork (has GRAND ID)
      arrange(grand_id) |>
      mutate(nidid = first(nidid), grand_id = first(grand_id),
             COMID = first(COMID), huc_12 = first(huc_12),
             pt_lon = first(pt_lon), pt_lat = first(pt_lat)),
    # 8. CASTAIC AND ALAMO. Separate facilities--should be split
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc1465") |> 
      mutate(grand_id = if_else(plant_name == "ALAMO", NA_real_, grand_id)) |> 
      mutate(RHPID = paste0(EIA_ID, "_", plant_name)),
    # 9. EDISON SAULT & ST MARYS FALLS. SAME RESOURCE, SPLIT OWNERSHIP: COMBINE.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc0265") |> 
      mutate(RHPID = "1751/1865_EDISON SAULT & ST MARYS FALLS"),
    # 10. DEERBORN AND GREAT FALLS. SAME RESOURCE: COMBINE.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc1323") |> 
      mutate(RHPID = "3256/3259_DEERBORN & GREAT FALLS"),
    # 11. SAN FRANCISQUITO 1 & 2. SAME RESOURCE: COMBINE.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc4103") |> 
      mutate(RHPID = "6479/6480_SAN FRANCISQUITO 1 & 2"),
    # 12. JEFFRIES & ST STEPHEN - SAME RESERVOIR, DIFFERENT OUTLET. SPLIT.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc0266") |> 
      mutate(RHPID = paste0(EIA_ID, "_", plant_name)),
    # 13. MCNARY AND MCNARY FISH. SAME RESOURCE: COMBINE.
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc0252") |> 
      mutate(RHPID = "3084/7511_MCNARY & MCNARY FISH"),
    # 14. SNOQUALMIE. SAME RESOURCE. COMBINE. 
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc1519") |> 
      mutate(RHPID = "3860/7867_SNOQUALMIE 1 & 2"),
    # 15. HOOVER NV and AZ. COMBINE. 
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc0025") |> 
      mutate(RHPID = "154/8902_HOOVER DAM (NV & AZ)"),
    # 16. GLENS FALLS. SAME RESOURCE: COMBINE. 
    target_plants_mapped_to_water_corrected_H |> 
      filter(COMPLXID == "hc1437") |> 
      mutate(RHPID = "54772/56936_SOUTH GLENS FALLS & GLENS FALLS")
    ) ->
    target_plants_mapped_to_water_corrected_fixed_water_duplicates

  
  target_plants_mapped_to_water_corrected_H |> 
    filter(!COMPLXID %in% target_plants_mapped_to_water_corrected_fixed_water_duplicates[["COMPLXID"]]) |> 
    mutate(RHPID = paste0(EIA_ID, "_", plant_name)) |> 
    bind_rows(target_plants_mapped_to_water_corrected_fixed_water_duplicates) |> 
    left_join(plant_dominant_modes, by = join_by(EIA_ID)) |> 
    select(EIA_ID, COMPLXID, RHPID, grand_id, nidid, COMID, huc_12, H_m,
           dominant_mode, pt_lon, pt_lat, usgs_gage) ->
    target_plants_mapped_to_water_corrected_RHPIDs

  return(target_plants_mapped_to_water_corrected_RHPIDs)

}


get_ResOps_releases <- function(target_plants_mapped_to_water,
                                ResOpsUSrelease_csv){

  target_plants_mapped_to_water |> 
    filter(!is.na(grand_id)) -> 
    target_plants_with_grand
  
  vroom(ResOpsUSrelease_csv,
        col_type = cols(date = col_date(),
                        .default = col_double())) ->
    resOpsrelease

  
  suppressWarnings(
    tibble(
      grand_id = names(resOpsrelease) |> as.integer()
    ) |> filter(grand_id %in% target_plants_with_grand[["grand_id"]]) |> 
      pull(grand_id) |> as.character() ->
      grand_to_map
  )
  
  resOpsrelease |> 
    select(date, one_of(grand_to_map)) |> 
    # correct for weird date duplication in ResOpsUS table
    unique() ->
    resOps_release_daily
  
  return(resOps_release_daily)

}

